#include "process.h"
#include "frame.h"
#include "symtab.h"
#include "symbol.h"
#include "core.h"
#include "coff.h"
#include "asm.h"
#include "dsp32asm.h"
#include "dsp32core.h"
SRCFILE("dsp32core.c")

int Dsp32Core::REG_FP()			{ return 13; }
int Dsp32Core::REG_AP()			{ return 13; }
int Dsp32Core::REG_SP()			{ return 14; }
int Dsp32Core::REG_PC()			{ return 27; }
int Dsp32Core::nregs()			{ return 34; }
int Dsp32Core::cntxtbytes()		{ return 33 << 2; }
int Dsp32Core::argdir()			{ return -1; }
int Dsp32Core::argstart()		{ return -8; }
char *Dsp32Core::dumppipeline()		{ return 0; }
long Dsp32Core::callingpc(long fp)	{ return peek(fp-4)->lng; }
long Dsp32Core::callingfp(long fp)	{ return peek(fp)->lng; }
long Dsp32Core::instrafterjsr()		{ return pc()+8; }
Asm *Dsp32Core::newAsm()		{ return new Dsp32Asm(dsp32c, this); }

void Dsp32Core::newSymTab(long reloc)
{
	_symtab = new CoffSymTab(this, stabfd, _symtab, reloc);
}

Dsp32Core::Dsp32Core()
{
	stackdir = GROWUP;
	memlayout = LSBFIRST;
	bptsize = 0;
}

char *Dsp32Core::pokedbl(long l,double d,int)
{
	long dspf = dtodsp(d);
	return poke(l, dspf, 4);
}

char *Dsp32Core::poke(long l,long d,int n)
{
	unsigned char raw[4];
	register unsigned char *r = raw;
	int i = n;
	while( i-- ) {
		*r++ = (unsigned char)d;
		d >>= 8;
	}
	return write( l, (char*)raw, n );
}

#define REGBIT(r) (1 << r)
long Dsp32Core::saved(Frame *f, int r, int)	/* ignore size */
{
	int reg;
	if (!(f->regsave & REGBIT(r)))
		return 0;
	long loc = f->regbase;
	for (reg = 5; reg < 13; reg++) {
		if (reg == r)
			return loc;
		if (f->regsave & REGBIT(reg))
			loc += dsp32c ? 4 : 2;
	}
	if (loc & 2)
		loc += 2;
	return (r == 25) ? loc : loc + 4;
}

char *Dsp32Core::regname(int r)
{
	static char *regnames[] = {
		"r0", "r1", "r2", "r3", "r4", "r5", "r6", "r7",
		"r8", "r9", "r10", "r11", "r12", "fp", "sp", "r15",
		"r16", "r17", "r18", "r19", "pin", "pout", "ivtp",
		"a0", "a1", "a2", "a3",
		"pc", "pcsh", "dauc", "ioc", "pcw", "npc", "ticks"
	};
	if (r >= 0 && r < nregs())
		return regnames[r];
	else
		return 0;
}

#define CYCLE 4
Cslfd *Dsp32Core::peek(long loc, Cslfd *fail)
{
	static i;
	static Cslfd *c;
	unsigned char raw[4];

	if( read(loc, (char*)raw, 4) )
		return Core::peek(loc, fail);
	if( !c ) c = new Cslfd[CYCLE];
	Cslfd *p = c+(i++,i%=CYCLE);
	p->chr = raw[0];
	p->sht = (raw[1]<<8) | raw[0];
	p->lng = (raw[3]<<24) | (raw[2]<<16) | (raw[1]<<8) | raw[0];
	p->flterr = 0;
	p->flt = p->dbl = dsptod(p->lng);
	return p;
}

const long	DSP_MASK1 = 0x7FFF0000,
		DSP_ADDSPSP = 0x15CE0000,
		DSP_MASK2 = 0xFFA0FFFD,
		DSP_MOVREGSPP = 0x1FA001D5,
		DSP_MOVA2SPP = 0x30400177,
		DSP_MOVA3SPP = 0x306001F7;
Frame Dsp32Core::frameabove(long _fp)
{
	int reg;
	Frame f(this);
	if( _fp ){
		f.pc = callingpc(_fp);
		f.fp = callingfp(_fp);
	} else {
		f.pc = pc();
		f.fp = fp();
	}
	f.ap = f.fp;
	f.regbase = f.fp + 4;
	Func *funcp = (Func*)_symtab->loctosym(U_FUNC, f.pc);
	if (funcp) {
		int faddr = (int)funcp->range.lo + 12;
		long inst = peek(faddr)->lng;
		if ((inst & DSP_MASK1) == DSP_ADDSPSP) {
			f.regbase += inst & 0xFFFF;
			faddr += 4;
			inst = peek(faddr)->lng;
		}
		if ((inst & DSP_MASK2) == DSP_MOVREGSPP &&
		    (reg = (int)(inst >> 16) & 0x1F) >= 5) {
			do {
				f.regsave |= REGBIT(reg);
				faddr += 4;
			} while (++reg <= 12);
			inst = peek(faddr)->lng;
		}
		if (inst == DSP_MOVA2SPP)
			f.regsave |= REGBIT(25) | REGBIT(26);
		else if (inst == DSP_MOVA3SPP)
			f.regsave |= REGBIT(26);
	}
	return f;
}

const long DSP_CALLMASK = 0xFC000000, DSP_CALL = 0x10000000;
const long DSP_CALL2MASK = 0xE0000000, DSP_CALL2 = 0xE0000000;
int Dsp32Core::atjsr(long pc)
{
	long inst = peek(pc)->lng;
	return ( (inst & DSP_CALLMASK) == DSP_CALL ||
		 (inst & DSP_CALL2MASK) == DSP_CALL2 );
}

const long	RMASK = 0x7FFF0000, RINSTR = 0x15C30000; // sp = r3 + const
int Dsp32Core::atreturn(long pc) { return (peek(pc)->lng & RMASK) == RINSTR; }

char *Dsp32Core::stepprolog()
{
	long instr;
	long pcs = pc();
	// push r18; push fp; fix fp
	char *error = step(pcs, pcs + 12);
	if (!error && (peek(pcs = pc())->lng & DSP_MASK1) == DSP_ADDSPSP) {
		error = step();	// sp += const
		pcs = pc();
	}
	if (!error && ((instr=peek(pcs)->lng) & DSP_MASK2) == DSP_MOVREGSPP) {
		int reg = (int)(instr >> 16) & 0x1F;
		if (reg >= 5 && reg <= 12) {
			 // sp++[r19] = reg
			error = step(pcs, pcs + 4 * (13 - reg));
			pcs = pc();
		}
	}
	if (!error && peek(pcs)->lng == DSP_MOVA2SPP) {
		error = step(pcs, pcs + 8);	// sp++ = a2; sp++ = a3
		pcs = pc();
	} else if (!error && peek(pcs)->lng == DSP_MOVA3SPP)
		 error = step();	// sp++ = a3 = a3
	return error;
}

char *Dsp32Core::docall(long addr, int numarg)
{
	const int CALL_SIZE=3,
		DSP32_CALL = 0x12800000,
		DSP32_FIXR18 = 0x16800000,
		DSP32C_CALL = 0xE0140000,
		DSP32C_FIXR18 = 0xC0140000,
		DSP32_SUBSP = 0x1A8E0000;
	long save[CALL_SIZE], newcode[CALL_SIZE];
	char *error = 0;
	int i;

	if (behavetype() == ACTIVE)
		return "process not stopped";
	long callstart = scratchaddr();
	if (dsp32c) {
		newcode[0] = DSP32C_CALL | (addr & 0xFFFF) |
			((addr & 0xFF0000) << 5);
		newcode[1] = DSP32C_FIXR18 | ((callstart + 8) & 0xFFFF) |
			(((callstart + 8) & 0xFF0000) << 5);
		newcode[2] = DSP32_SUBSP | (numarg << 2) | 0x80000000;
	} else {
		newcode[0] = DSP32_CALL | (addr & 0xFFFF);
		newcode[1] = DSP32_FIXR18 | ((callstart + 8) & 0xFFFF);
		newcode[2] = DSP32_SUBSP | (numarg << 2);
	}
	if (error = read(callstart, (char*)save, sizeof(save)))
		return error;
	for (i = 0; i < CALL_SIZE; ++i)
		if (error = poke(callstart+(i << 2), newcode[i], 4))
			return error;
	if ((error = regpoke(REG_PC(), callstart))
	 || (error = regpoke(32, callstart + 4)) // npc
	 || (error = step(callstart, callstart+sizeof(newcode)))
	 || (error = dumppipeline())
	 || (error = write(callstart, (char*)save, sizeof(save))))
		return error;
	Func *funcp = (Func*)_symtab->loctosym(U_FUNC, addr);
	if (funcp && funcp->type.decref()->isreal())
		floatcall = 1;
	else
		floatcall = 0;
	return 0;
}

long Dsp32Core::apforcall(int argbytes)
{
	long ap = sp() + argbytes;
	regpoke(REG_SP(), ap);
	return ap + 4;
}

long Dsp32Core::returnregloc()
{
	if (floatcall)
		return regloc(23);
	else
		return regloc(1);
}

const long	FRACTMASK=0x7fffffL,
		ONE=0x800000L,
		MASKEXP=0xffL;
long Dsp32Core::dtodsp(double a) 
{
	long i, j, val, sign;

	if (a == 0.0)
		return 0;
	if(a < 0.0) {
		sign = -1;
		a = -a;
	} else
		sign = 1;
	if(a >= 2.0)
		for(i = 0; a >= 2.0; i++)
			a /= 2.0;
	else
		for(i = 0; a < 1.0; i--)
			a *= 2.0;
	j = (long)(a * (double)ONE + .5);
	if (sign == -1 && j == ONE) {
	    j *= 2;
	    i--;
	}
	if (sign == 1 && j == (ONE * 2)) {
	    j /= 2;
	    i++;
	}
	j *= sign;
	if (sign == -1)
	    val = 1 << 31;
	else
	    val = 0;
	i += 128;
	val |= i & MASKEXP;
	j &= FRACTMASK;
	val |= j << 8;
	return val;
}

double Dsp32Core::dsptod(long l)
{
	long exp = (MASKEXP & l) - 128;
	l >>= 8;
	long sign = l & ONE;
	l &= FRACTMASK;
	if(exp == -128)
		return 0.0;
	if (!sign)
		l += ONE;
	else 
		l += ONE * -2;
	double retval = 1.0;
	for( ; exp > 0; exp--)
		retval *= 2.0;
	for( ; exp < 0; exp++)
		retval /= 2.0;
	retval *= (double)l * (1.0/(double)(1L<<23));
	return retval;
}

Context	*Dsp32Core::newContext()	{ return new Dsp32Context(this); }
Dsp32Context::Dsp32Context(Dsp32Core *c):Context(c) { dspcore = c; }

void Dsp32Context::save()
{
	if( dspcore->behavetype() == ACTIVE )
		error = "context save: process not stopped";
	if (!error) error = dspcore->dumppipeline();
	if (!error) Context::save();
}
